define(['angular', 'app', 'moment', 'PatientFHIRService'], function (angular, app, moment) {
    'use strict';

    app.service('TrackersFHIRService', function ($q, BaseFHIRService, PatientFHIRService) {

        return angular.extend({}, angular.copy(BaseFHIRService), {
            type: 'Observation',
            dataPointers: {},



            save: function (ngModel) {
                ngModel.effectiveDateTime = this.buildEffectiveDateTime(ngModel);

                return PatientFHIRService.getPatientContext().then(function(patientContext){
                    if (patientContext) {
                        ngModel.subject.reference = 'Patient/' + patientContext.id;

                        var requestType = ngModel.id ? 'update': 'create';
                        var resource = this.buildResource(ngModel);


                        return this[requestType](resource).then(function (response) {
                            // Workaround because the FHIR server doesn't allow querying immediately after creating a resource
                            setTimeout(function () {

                                this.fetch().then(function () {
                                    this.latest = this.dataPointers.list[0];
                                    return response;
                                }.bind(this));

                            }.bind(this), 1000);

                        }.bind(this));

                    };

                }.bind(this));


            },

            buildEffectiveDateTime: function (ngModel) {
                var combinedDateTimeFormat = 'MM/DD/YYYY hh:mm A';
                var combinedDateTime = new moment(ngModel.inputEntryDate + ' ' + ngModel.inputEntryTime, combinedDateTimeFormat);

                return combinedDateTime.format('YYYY-MM-DDTHH:mm:ssZ');
            },

            buildResource: function (ngModel) {
                angular.extend(ngModel, {
                    resourceType: this.type,
                    effectiveDateTime: this.buildEffectiveDateTime(ngModel),
                    inputEntryDate: undefined,
                    inputEntryTime: undefined
                });

                if (ngModel.comments === "") {
                    // empty comments are not allowed to be saved
                    delete ngModel.comments;
                }

                if (ngModel.valueQuantity) {
                    // convert string numbers back to regular int
                    ngModel.valueQuantity.value = parseInt(ngModel.valueQuantity.value);
                }

                return ngModel;
            },

            create: function (resource) {
                return this.checkForSameDate(resource.effectiveDateTime).then(function () {

                    return BaseFHIRService.create.call(this, resource);
                }.bind(this));
            },

            checkForSameDate: function (date) {
                var errorsObj = {
                    'objectType': 'ValidationErrors',
                    errors: [
                        {
                            errorMessage: 'An entry has already been saved with the same date and time. Please change your date and/or time and save again.'
                        }
                    ]
                };

                return $q(function (resolve, reject) {
                    this.search({
                        code: this.code,
                        date: date
                    }).then(function (response) {
                        var count = response.getTotal();

                        if (count) {
                            reject(errorsObj);
                        } else {
                            resolve();
                        }
                    }, function (response) {
                        reject(response);
                    });
                }.bind(this));
            },

            delete: function (ngModel) {
                return BaseFHIRService.delete.call(this, ngModel.id).then(function (response) {
                    var urlSections = response.config.url.split('/');

                    this.removeFromLocalList(urlSections[urlSections.length - 1]);
                    this.latest = this.dataPointers.list[0];

                    return response;
                }.bind(this));
            },

            removeFromLocalList: function (id) {
                var item = this.localGetById(id);
                var index = this.dataPointers.list.indexOf(item);

                if (index !== -1) {
                    this.dataPointers.list.splice(index, 1);
                }
            },

            getLatest: function () {
                return PatientFHIRService.getPatientContext().then(function (patientContext){
                    if(patientContext) {
                        return this.search({
                            _count: 1,
                            code: this.code,
                            '_sort:desc': 'date',
                            patient: patientContext.id
                        }).then(function (response) {
                            this.latest = response.getItems()[0];
                        }.bind(this));
                    }
                }.bind(this));
            },

            localGetById: function (id) {
                var list = this.dataPointers.list || [];

                var item = this.findInArray(list, function (item) {
                    return item.id === id;
                });

                if (item) {
                    this.transformItem(item);
                }

                return item;
            },

            transformItem: function (item) {
                angular.extend(item, {
                    inputEntryDate: this.getDateFromDateTime(item.effectiveDateTime),
                    inputEntryTime: this.getTimeFromDateTime(item.effectiveDateTime)
                });
            },

            createEmpty: function () {
                var today = moment();

                return {
                    status: 'preliminary',
                    inputEntryDate: this.getDateFromDateTime(today),
                    inputEntryTime: this.getTimeFromDateTime(today),
                    comments: '',
                    subject: {}
                };
            },

            getDateFromDateTime: function (dateTime, format) {
                format = format || 'MM/DD/YYYY'

                return moment(new Date(dateTime)).format(format);
            },

            getTimeFromDateTime: function (dateTime) {
                return moment(new Date(dateTime)).format('hh:mm A');
            },

            fetch: function (filtersObj) {
                filtersObj = filtersObj || {};


                return PatientFHIRService.getPatientContext().then(function(patientContext){
                    if(patientContext) {
                        var filters = angular.copy(filtersObj);
                        var dateFormat = 'YYYY-MM-DD';

                        angular.extend(filters, {
                            code: this.code,
                            '_sort:desc': 'date',
                            patient: patientContext.id,
                            startDate: filtersObj.startDate ? this.getDateFromDateTime(filtersObj.startDate, dateFormat) : moment().subtract(1, 'y').format(dateFormat),
                            endDate: filtersObj.endDate ? this.getDateFromDateTime(filtersObj.endDate, dateFormat) : moment().format(dateFormat)



                        });

                        return this.search(filters).then(function (response) {
                            return response.getAllItems().then(function (response) {
                                this.dataPointers.list = response;

                                return;
                            }.bind(this));
                        }.bind(this));
                    }
                }.bind(this));
            }
        });
    });

});